<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;

use App\Tipo;
use App\Solicitud;
use App\Respuesta;
use App\Notificacion;

use App\Mail\MailRespuestaServicio;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Exception;
use DB;
use Illuminate\Support\Carbon;

class RespuestaServicioCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'correo:servicio';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Envía un correo de respuesta al vecino para aquellos servicios que trabajan con respuesta automatica';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $sent_emails    =   0;
        $task_estado    =   false;
        $start_time     =   microtime(true);
        DB::connection()->enableQueryLog();

        // Se obtiene los servicios que trabajan con respuesta automatica
        $data   =   Tipo::
                    join('gsut_respuesta_servicio as rs','rs.id_tipo','=','gsut_tipo.id')
                    ->join('gsut_solicitud as s','s.id_tipo','=','gsut_tipo.id')
                    ->join('gsut_vecino as vecino','vecino.nr_rut','=','s.nr_rut_vecino')
                    ->select('s.id as nr_solicitud',
                        's.nr_ticket',
                        DB::raw('concat("Solicitud de Atención Vecino - Ticket #",s.nr_ticket) as str_asunto'),
                        DB::raw('(timestampdiff(day, s.fc_creacion, now())) as diff_day'),
                        DB::raw('(timestampdiff(hour, s.fc_creacion, now())) as diff_hour'),
                        DB::raw('(timestampdiff(minute, s.fc_creacion, now())) as diff_minute'),
                        DB::raw('(case when ((timestampdiff(minute, s.fc_creacion, convert_tz(now(),"UTC","America/Santiago"))) > (60*rs.nr_tiempo)) then 1 else 0 end) as tp_notificar'),
                        'vecino.nr_rut',
                        'vecino.nm_vecino',
                        'vecino.nm_mail',
                        'rs.str_respuesta',
                        'rs.str_respuesta_html')
                    ->where('gsut_tipo.tp_activo','=','1')
                    ->where('gsut_tipo.tp_respuesta','=','1')
                    ->where('rs.tp_activo','=','1')
                    ->where('s.tp_activo','=','1')
                    ->where('s.id_estado','=','3')
                    ->where('s.tp_notificado','=','0')
                    ->orderBy('gsut_tipo.id','ASC')
                    ->orderBy('s.id','ASC')
                    ->get();

        if($data->count())
        {
            // Se obtienen los ticket a los cuales se debe notificar
            $ticket         =   $data->where('tp_notificar','=','1');
            $nr_intentos    =   3;
            $nr_espera      =   1;
            $tp_enviado     =   0;
            $nr_delay       =   2;

            foreach ($ticket as $key => $value) {
                $fc_actual  =   Date("Y-m-d H:i:s");

                try {
                    try {
                        Mail::to($value->nm_mail)
                            ->send(new MailRespuestaServicio(
                                $value->str_asunto, 
                                $value->str_respuesta_html, 
                                $value->nm_vecino, 
                                $value
                            ));

                        $tp_enviado     =   1;
                    } catch (Exception $e) {
                        retry($nr_intentos, function () use ($value) {
                            Mail::to($value->nm_mail)->send(new MailRespuestaServicio(
                                $value->str_asunto, 
                                $value->str_respuesta_html, 
                                $value->nm_vecino, 
                                $value
                            )); 
                        }, $nr_espera);

                        // Si después de reintentar el envío sigue habiendo un error, llevar un registro de los correos que fallaron
                        if ($e instanceof Swift_TransportException) {
                            Log::error('Error al enviar correo a ' . $value->nm_mail . ': ' . $e->getMessage());
                            $tp_enviado     =   0;

                            // Se registra respuesta
                            $respuesta  =   [
                                'id_solicitud'          =>  $value->nr_solicitud,
                                'id_autor'              =>  0, // Sistema
                                'fc_ingreso'            =>  $fc_actual,
                                'str_observacion'       =>  'No es posible enviar la Respuesta Automatica del Servicio al mail "'.$value->nm_mail.'"',
                                'str_observacion_html'  =>  '<p>No es posible enviar la Respuesta Automatica del Servicio al mail "'.$value->nm_mail.'"</p>',
                                'id_historia'           =>  4,
                                'str_ruta_adjunto'      =>  null,
                                'tp_respuesta'          =>  13,
                                'tp_validado'           =>  0,
                                'nr_fuente'             =>  0,
                                'tp_activo'             =>  1,
                            ];

                            Respuesta::create($respuesta);

                            // Se registra notificcion en la tabla
                            $notificacion   =   [
                                'nr_solicitud'      =>  $value->nr_solicitud,
                                'nr_ticket'         =>  $value->nr_ticket,
                                'desde'             =>  'atencionvecino@sanmiguel.cl',
                                'para'              =>  json_encode($value->nm_mail),
                                'tp_notificacion'   =>  1002,
                                'str_asunto'        =>  $value->str_asunto,
                                'tp_estado'         =>  0,
                                'str_exception'     =>  $e->getMessage(),
                            ];

                            Notificacion::create($notificacion);
                        }
                        else
                            $tp_enviado     =   1;
                    }
                        
                    if($tp_enviado == 1)
                    {
                        // Se registra respuesta
                        $respuesta  =   [
                            'id_solicitud'          =>  $value->nr_solicitud,
                            'id_autor'              =>  0, // Sistema
                            'fc_ingreso'            =>  $fc_actual,
                            'str_observacion'       =>  $value->str_respuesta,
                            'str_observacion_html'  =>  $value->str_respuesta_html,
                            'id_historia'           =>  4,
                            'str_ruta_adjunto'      =>  null,
                            'tp_respuesta'          =>  12,
                            'tp_validado'           =>  0,
                            'nr_fuente'             =>  0,
                            'tp_activo'             =>  1,
                        ];

                        Respuesta::create($respuesta);
                        
                        // Marca para saber si se notifico al vecino
                        Solicitud::where('id','=',$value->nr_solicitud)
                        ->update([
                            'tp_notificado'     =>  1,
                        ]);

                        // Se registra notificcion en la tabla
                        $notificacion   =   [
                            'nr_solicitud'      =>  $value->nr_solicitud,
                            'nr_ticket'         =>  $value->nr_ticket,
                            'desde'             =>  'atencionvecino@sanmiguel.cl',
                            'para'              =>  json_encode($value->nm_mail),
                            'tp_notificacion'   =>  1001,
                            'str_asunto'        =>  $value->str_asunto,
                        ];

                        Notificacion::create($notificacion);
                    }
                    
                    $task_estado    =   true;
                    $sent_emails++;
                    sleep($nr_delay);
                } catch (Exception $e) {
                    $task_estado    =   false;
                    \Log::error(__METHOD__ . " Tarea Programada - Notificaciones al Vecino - Respuesta del Servicio : Error " . 
                        'start_time : ' . $start_time . 
                        'end_time : ' . Carbon::now() . 
                        'exception : ' . $e->getMessage() . 
                        'toString : ' . $e->__toString());
                }
            }

            $end_time       =   microtime(true);
            $total_time     =   $end_time - $start_time;

            DB::table('task_log')->insert([
                'task_name'     =>  'Notificaciones al Vecino - Respuesta del Servicio',
                'start_time'    =>  date('Y-m-d H:i:s', $start_time),
                'end_time'      =>  date('Y-m-d H:i:s', $end_time),
                'duration'      =>  $total_time,
                'success'       =>  $task_estado,
                'message'       =>  ($task_estado) ? 'Se han enviado "' . $sent_emails . '" notificaciones correctamente' : 
                                                    'Ha ocurrido un problema en la ejecucion de la tarea',
            ]);
        }
    }
}
