<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Vecino;
use App\Solicitud;
use App\Historial;
use App\Respuesta;
use App\Satisfaccion;
use App\SolicitudTerritorial;

use App\Http\Controllers\UtilController;
use App\Http\Controllers\BusController;
use App\Http\Requests\SolicitudRequest;

use DB;
use Validator;
use GuzzleHttp\Client;
use Freshwork\ChileanBundle\Rut;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class GsutVecinoController extends Controller
{
    //------------------------------------------------------------
    public function logoutVecino()
    {
        /*
        $data   =   (Object) [
                        'rut'       =>  '17105531-8',
                        'nombre'    =>  'Noel',
                        'apellido_paterno'  =>  'Gutierrez',
                        'apellido_materno'  =>  'Collinao',
                    ];

        Session::put('clave-unica',$data);
        */

        if(request()->logout){
            Session::forget('clave-unica');
            Session::forget('nr_rut');
            Session::forget('info_vecino');
            Session::flush();
        }

        return view('atencion_vecino.clave_unica');
    }
    //------------------------------------------------------------
    public function page($tp_destino = 'consulta')
    {
        $tp_msj_cu  =   env('TP_MSJ_CLAVE_UNICA',0);

        if(!Session::get('clave-unica')){
            return view('atencion_vecino.clave_unica', ['tp_msj_clave_unica' => $tp_msj_cu]);
        }

        $destino    =   '';
        $nr_rut     =   Session::get('clave-unica')->rut;

        if(Rut::parse($nr_rut)->quiet()->validate())
        {
            $data       =   (new GsutVecinoController)->getVecino($nr_rut);
            Session::put('nr_rut', $nr_rut);
            Session::put('info_vecino', $data);
        }

        $url_video  =   'https://assets-lascondes.s3.us-east-1.amazonaws.com/video/atencion_vecino/busqueda_de_servicios.mp4';
        $client     =   new Client();
        $tp_activo  =   false;

        try {
            $response   =   $client->head($url_video);
            $tp_activo  =   $response->getStatusCode() === 200;
        } catch (\Exception $e) {
            $tp_activo  =   false;
        }


        if($tp_destino === 'inicio')
            $destino = 'atencion_vecino.inicio_vecino';

        if($tp_destino == 'consulta')
            $destino = 'atencion_vecino.ingreso_solicitud';

        if($tp_destino === 'solicitud')
            $destino = 'atencion_vecino.bandeja_vecino';

        if($tp_destino === 'perfil')
            $destino = 'atencion_vecino.perfil_vecino';

        if($tp_destino === 'logout')
        {
            Session::forget('clave-unica');
            Session::forget('nr_rut');
            Session::forget('info_vecino');
            Session::flush();

            $destino    =   'atencion_vecino.clave_unica';
        }

        return view($destino, ['tp_msj_clave_unica' => $tp_msj_cu])
                ->with(['url_video'=>$url_video, 'tp_video'=>$tp_activo]);
    }
    //------------------------------------------------------------
    public function pageIngresoDinamico()
    {
        $url_video  =   'https://assets-lascondes.s3.us-east-1.amazonaws.com/video/atencion_vecino/busqueda_de_servicios.mp4';
        $client     =   new Client();
        $tp_activo  =   false;

        try {
            $response   =   $client->head($url_video);
            $tp_activo  =   $response->getStatusCode() === 200;
        } catch (\Exception $e) {
            $tp_activo  =   false;
        }

        return view('atencion_vecino.ingreso_vecino_dinamico')
                ->with(['url_video'=>$url_video, 'tp_video'=>$tp_activo]);
    }
    //------------------------------------------------------------
    public function pageIngreso()
    {
        $url_video  =   'https://assets-lascondes.s3.us-east-1.amazonaws.com/video/atencion_vecino/busqueda_de_servicios.mp4';
        $client     =   new Client();
        $tp_activo  =   false;

        try {
            $response   =   $client->head($url_video);
            $tp_activo  =   $response->getStatusCode() === 200;
        } catch (\Exception $e) {
            $tp_activo  =   false;
        }

        return view('atencion_vecino.ingreso_vecino')
                ->with(['url_video'=>$url_video, 'tp_video'=>$tp_activo]);
    }
    //------------------------------------------------------------
    public function pageSumiderosAgualluvias() // Servicio de Sumideros de Aguas Lluvias
    {
        $url_video  =   'https://assets-lascondes.s3.us-east-1.amazonaws.com/video/atencion_vecino/busqueda_de_servicios.mp4';
        $client     =   new Client();
        $tp_activo  =   false;

        try {
            $response   =   $client->head($url_video);
            $tp_activo  =   $response->getStatusCode() === 200;
        } catch (\Exception $e) {
            $tp_activo  =   false;
        }

        return view('servicio_especifico.ingreso_servicio_sumideros')
                ->with(['url_video'=>$url_video, 'tp_video'=>$tp_activo]);
    }
    //------------------------------------------------------------
    public function pageAudiencia()
    {
        $url_video  =   'https://assets-lascondes.s3.us-east-1.amazonaws.com/video/atencion_vecino/busqueda_de_servicios.mp4';
        $client     =   new Client();
        $tp_activo  =   false;

        try {
            $response   =   $client->head($url_video);
            $tp_activo  =   $response->getStatusCode() === 200;
        } catch (\Exception $e) {
            $tp_activo  =   false;
        }

        return view('audiencias.ingreso_audiencia')
                ->with(['url_video'=>$url_video, 'tp_video'=>$tp_activo]);
    }
    //------------------------------------------------------------
    public function pageRedesSociales()
    {
        $url_video  =   'https://assets-lascondes.s3.us-east-1.amazonaws.com/video/atencion_vecino/busqueda_de_servicios.mp4';
        $client     =   new Client();
        $tp_activo  =   false;

        try {
            $response   =   $client->head($url_video);
            $tp_activo  =   $response->getStatusCode() === 200;
        } catch (\Exception $e) {
            $tp_activo  =   false;
        }

        return view('redes_sociales.ingreso_rrss')
                ->with(['url_video'=>$url_video, 'tp_video'=>$tp_activo]);
    }
    //------------------------------------------------------------
    public function pageConsulta()
    {
        if(request()->logout){
            Session::forget('clave-unica');
        }

        if(Session::get('clave-unica')){
            return view('atencion_vecino.consulta_vecino');
        }else{
            return view('atencion_vecino.clave_unica');
        }
    }
    //------------------------------------------------------------
    public function storeVecino(Request $request)
    {
        $valida     =   $this->validarDatos($request);

        if($valida['error'] == 1)
            return $valida;

        if(Rut::parse($request->rut_vecino)->quiet()->validate())
            return self::setRegistrarVecino($request);
        else
            return $valida =   [
                'error'     =>  1,
                'existe'    =>  0,
                'tp_icon'   =>  'warning',
                'title'     =>  'Registro de Vecino',
                'msj'       =>  'El Rut ingresado no es valido, favor verificar.'
            ];
    }
    //------------------------------------------------------------
    public static function getVecino($nr_rut)
    {
        $valido     =   Rut::parse($nr_rut)->quiet()->validate();
        $tramites   =   0;
        $vecino     =   null;

        if(!$valido)
            return ['error' => 1, 'vecino'=> $vecino,'tramites'=>$tramites];

        $nr_rut     =   Rut::parse($nr_rut)->format(Rut::FORMAT_WITH_DASH);

        $vecino     =   Vecino::
                        select(
                            'nr_rut', 
                            'ap_paterno',
                            'ap_materno',
                            'nm_vecino',
                            'tp_direccion',
                            'str_aclaratoria',
                            'nr_telefono',
                            'nr_prefijo',
                            'nm_mail',
                            'id_region',
                            'id_comuna',
                            'id as id_ubicacion',
                            'nr_calle',
                            'nm_calle',
                            'nm_ubicacion',
                            'str_latitud',
                            'str_longitud',
                            DB::raw('(case when id_comuna=271 then 1 else 0 end) as tp_comuna'),
                            'id_unidadvecinal as cod_unidadvecinal',
                            'id_macroterritorio as cod_macroterritorio')
                        ->with('region','comuna')
                        ->where('nr_rut','=',$nr_rut);

        if($vecino->count())
        {
            $vecino     =  $vecino->first();
            $tramites   =   0;
        }
        else
        {
            $tramites   =   1;

            if(Session::get('clave-unica')) {
                $vecino     =   (Object) [
                                    'nr_rut'        =>  Session::get('clave-unica')->rut, 
                                    'ap_paterno'    =>  Session::get('clave-unica')->apellido_paterno,
                                    'ap_materno'    =>  Session::get('clave-unica')->apellido_materno,
                                    'nm_vecino'     =>  Session::get('clave-unica')->nombre,
                                    'correo'        =>  '',
                                    'nr_telefono'   =>  '',
                                ];
            }
            else
            {
                $datos  =   RestController::consultaBusDatosVecino($nr_rut);

                if(isset($datos))
                {
                    $vecino     =   (Object) [
                                        'nr_rut'        =>  $nr_rut,
                                        'ap_paterno'    =>  $datos->ap_paterno,
                                        'ap_materno'    =>  $datos->ap_materno,
                                        'nm_vecino'     =>  $datos->nombre,
                                        'correo'        =>  $datos->email,
                                        'nr_telefono'   =>  '',
                                    ];
                }
                else
                {
                    $vecino     =   (Object) [
                                        'nr_rut'        =>  $nr_rut, 
                                        'ap_paterno'    =>  '',
                                        'ap_materno'    =>  '',
                                        'nm_vecino'     =>  '',
                                        'correo'        =>  '',
                                        'nr_telefono'   =>  '',
                                    ];
                }
            }
        }


        return ['error' => 0, 'vecino'=> $vecino,'tramites'=>$tramites];
    }
    //------------------------------------------------------------
    public static function getVecinoTramite($rut)
    {
        $base_uri   = 'https://solicitudes.lascondes.cl/get/vecino/'.$rut;
        $client     =  new Client(['base_uri' => $base_uri]);
        $data       = [];
        $response   = [];
        $vecino     = null;

        \Log::debug(__METHOD__ . " GEt Vecino : " . $base_uri);

        $options = ['verify' => false];

        try {
            $response   = $client->request('GET', "",$options);
            $data       = json_decode($response->getBody()->getContents());

            if($data)
                $vecino     =   $data->vecino;

        } catch (RequestException $ex){
            if ($ex->getResponse()->getStatusCode() == '400') {
                \Log::debug(__METHOD__ . $ex->getMessage());
            }
        }

        return $vecino;
    }
    //------------------------------------------------------------
    public function searchSolicitud(Request $request)
    {
        //$str_buscar     =   Rut::parse($request->str_buscar)->format(Rut::FORMAT_WITH_DASH);
        //dd(Session::get('clave-unica'));
        $str_buscar     =   Session::get('clave-unica')->rut;
        $fl_mostrar     =   $request->fl_mostrar;
        $page           =   $request->page;

        \Log::debug(__METHOD__ . " Consulta de Solicitud - Rut : " . Session::get('clave-unica')->rut);

        $data           =   Solicitud:: 
                            join('gsut_prioridad as prioridad','prioridad.id','=','gsut_solicitud.id_prioridad')
                            //->join('gsut_motivo as motivo','motivo.id','=','gsut_solicitud.id_motivo')
                            ->join('gsut_tipo as tipo','tipo.id','=','gsut_solicitud.id_tipo')
                            ->join('gsut_estado as estado','estado.id','=','gsut_solicitud.id_estado')
                            ->join('gsut_direccion as dir_destino','dir_destino.id','=','gsut_solicitud.id_direccion_destino')
                            ->join('gsut_depto as depto_destino','depto_destino.id','=','gsut_solicitud.id_depto_destino')
                            ->leftjoin('gsut_macroterritorio as mt','mt.id','=','gsut_solicitud.id_macroterritorio')
                            ->leftjoin('gsut_unidadvecinal as uv','uv.id','=','gsut_solicitud.id_unidadvecinal')
                            ->join('gsut_tipo_ingreso as ti','ti.id','=','gsut_solicitud.tp_ingreso')
                            ->select('gsut_solicitud.id',
                                'gsut_solicitud.nr_ticket',
                                'gsut_solicitud.id_municipio',
                                'gsut_solicitud.id_direccion',
                                'gsut_solicitud.id_depto',
                                'gsut_solicitud.id_tipo',
                                'gsut_solicitud.nm_tipo',
                                'gsut_solicitud.id_prioridad',
                                //'gsut_solicitud.id_motivo',
                                //'motivo.nm_motivo',
                                'gsut_solicitud.id_direccion_destino',
                                'dir_destino.nm_direccion as nm_direccion_destino',
                                'gsut_solicitud.id_depto_destino',
                                'depto_destino.nm_depto as nm_depto_destino',
                                'gsut_solicitud.id_ubicacion',
                                'gsut_solicitud.id_macroterritorio',
                                'mt.cod_territorio',
                                'gsut_solicitud.id_unidadvecinal',
                                'uv.cod_unidadvecinal',
                                'uv.nm_unidadvecinal',
                                'gsut_solicitud.tp_direccion',
                                'gsut_solicitud.nr_direccion',
                                'gsut_solicitud.nm_direccion',
                                'gsut_solicitud.nm_ubicacion',
                                'gsut_solicitud.str_latitud',
                                'gsut_solicitud.str_longitud',
                                'gsut_solicitud.nr_rut_vecino',
                                'gsut_solicitud.str_observacion',
                                'gsut_solicitud.str_observacion_html as str_solicitud_html',
                                'gsut_solicitud.str_aclaratoria',
                                'gsut_solicitud.tp_visado_mt',
                                'gsut_solicitud.tp_rechazado',
                                'gsut_solicitud.id_estado',
                                'estado.nm_color',
                                //'gsut_solicitud.contexto_municipal',
                                'gsut_solicitud.tp_ingreso',
                                'ti.nm_ingreso',
                                'gsut_solicitud.id_autor',
                                'gsut_solicitud.tp_activo',
                                'gsut_solicitud.tp_feedback',
                                'gsut_solicitud.tp_recepcion',
                                'gsut_solicitud.tp_satisfaccion',
                                'gsut_solicitud.str_feedback',
                                'gsut_solicitud.tp_feedback_vecino',
                                'gsut_solicitud.fc_feedback_vecino',
                                'gsut_solicitud.tp_satisfaccion_vecino',
                                'gsut_solicitud.str_feedback_vecino',
                                'gsut_solicitud.fc_creacion as fc_ingreso',
                                'gsut_solicitud.fc_publicado',
                                'gsut_solicitud.created_at',
                                'gsut_solicitud.updated_at as fc_tramitacion',
                                DB::raw('0 as territorial'))
                            ->selectRaw("(case 
                                            when estado.id = 1 then 'Ingresada'
                                            when estado.id = 2 then 'En Revisión'
                                            when estado.id = 3 then 'En Revisión'
                                            when estado.id = 4 then 'En Revisión'
                                            when (estado.id = 5 && gsut_solicitud.tp_feedback_vecino = 0) then 'Cerrado sin Calificación'
                                            when (estado.id = 5 && gsut_solicitud.tp_feedback_vecino = 1) then 'Cerrado con Calificación'
                                            when (estado.id = 6 && gsut_solicitud.tp_feedback_vecino = 0) then 'Cerrado sin Calificación'
                                            when (estado.id = 6 && gsut_solicitud.tp_feedback_vecino = 1) then 'Cerrado con Calificación'
                                            when estado.id = 7 then 'Anulada'
                                            when estado.id = 8 then 'Derivado'
                                            when estado.id = 9 then 'Derivado'
                                            when estado.id = 10 then 'Derivado'
                                            end) as nm_estado") 
                            ->where('gsut_solicitud.nr_rut_vecino','=',$str_buscar)
                            ->where('gsut_solicitud.tp_activo','=',1)
                            ->orderBy('gsut_solicitud.created_at','DESC');

        /*---------- Inicio - Filtro ----------*/
            if($fl_mostrar != 0)
            {
                if($fl_mostrar == 3)
                    $data   =   $data->whereIn('gsut_solicitud.id_estado',[1,2,3,4]);
                if($fl_mostrar == 5)
                    $data   =   $data->whereIn('gsut_solicitud.id_estado',[5,6]);
                if($fl_mostrar != 3 && $fl_mostrar != 5)
                    $data   =   $data->where('gsut_solicitud.id_estado','=',$fl_mostrar);
            }
        /*---------- Termino - Filtro ----------*/
        $data   =   $data->groupBy('gsut_solicitud.id');

        $data   =   $data->paginate(20);

        $data->getCollection()->transform(function ($value) {
            $value->tiempo_tramitado    =   UtilController::getTiempoTramitacion($value->fc_ingreso, $value->fc_publicado, $value->id_estado);
            $value->tiene_pp            =   (UtilController::getCantidadRespuestas(2, $value->id) != 0) ? 'SI' : 'NO'; // Tiene Publicaciones Parciales                
            return $value;
        });

        return [
            'pagination' => [
                'total'        => $data->total(),
                'current_page' => $data->currentPage(),
                'per_page'     => $data->perPage(),
                'last_page'    => $data->lastPage(),
                'from'         => $data->firstItem(),
                'to'           => $data->lastItem(),
            ],
            'info'             => $data,
            'rut_vecino'       => $str_buscar,
        ];
    }
    //------------------------------------------------------------
    public function getRespuestas($id_solicitud)
    {
        $data   =   Respuesta::
                    select('gsut_respuesta.id',
                        'gsut_respuesta.id_solicitud',
                        'gsut_respuesta.id_autor',
                        'gsut_respuesta.fc_ingreso',
                        'gsut_respuesta.str_observacion',
                        'gsut_respuesta.str_observacion_html',
                        'gsut_respuesta.id_historia',
                        'gsut_respuesta.str_ruta_adjunto',
                        'depto.nm_depto',
                        'dir.nm_direccion',
                        'gsut_respuesta.tp_respuesta')
                    ->selectRaw('DATE_FORMAT(gsut_respuesta.fc_ingreso,"%d-%m-%Y %H:%i") as fc_ingreso')
                    ->selectRaw("(case 
                        when gsut_respuesta.tp_respuesta = 1 then 'Publicación Definitiva' 
                        when gsut_respuesta.tp_respuesta = 2 then 'Publicación Parcial' 
                        when gsut_respuesta.tp_respuesta = 3 then 'Pre-Publicación Definitiva' 
                        when gsut_respuesta.tp_respuesta = 4 then 'Pre-Publicación Parcial' 
                        when gsut_respuesta.tp_respuesta = 5 then 'Pre-Publicación Definitiva Rechazada' 
                        when gsut_respuesta.tp_respuesta = 6 then 'Pre-Publicación Parcial Rechazada' 
                        when gsut_respuesta.tp_respuesta = 7 then 'Expediente' 
                        when gsut_respuesta.tp_respuesta = 8 then 'VB Jefatura' 
                        when gsut_respuesta.tp_respuesta = 9 then 'VB Director' 
                        when gsut_respuesta.tp_respuesta = 10 then 'Rechazo VB Jefatura' 
                        when gsut_respuesta.tp_respuesta = 11 then 'Rechazo VB Director' 
                        when gsut_respuesta.tp_respuesta = 12 then 'Respuesta Automática Enviada' 
                        when gsut_respuesta.tp_respuesta = 13 then 'Respuesta Automática Fallida' 
                        else 'Respuesta Interna' end) as nm_respuesta")
                    ->selectRaw('CONCAT(nombre," ",ap_paterno) as usuario')
                    ->join('users as autor','autor.id','=','gsut_respuesta.id_autor')
                    ->join('gsut_direccion as dir','dir.id','=','autor.id_direccion')
                    ->join('gsut_depto as depto','depto.id','=','autor.id_depto')
                    ->with('archivos')
                    ->where('id_solicitud','=',$id_solicitud)
                    ->orderBy('gsut_respuesta.created_at','asc')
                    ->whereIn('gsut_respuesta.tp_respuesta',[1,2])
                    ->where('gsut_respuesta.tp_activo','=',1)
                    ->get();

        return $data;
    }
    //------------------------------------------------------------
    public function getSatisfaccion($id_solicitud)
    {
        $info = Solicitud::select('tp_satisfaccion_vecino')->where('id','=',$id_solicitud)->first();
        
         if($info->tp_satisfaccion_vecino)
             {
                 $data = Satisfaccion::select('name')->where('id','=',$info->tp_satisfaccion_vecino)->first();
                 return ['value'=>$info->tp_satisfaccion_vecino,'label'=>$data->name];
             }
        return ['value'=>0,'label'=>'Sin feedback'];
    }
    //------------------------------------------------------------
    public function historialSolicitud(Request $request)
    {
        $id_solicitud       =   $request->id_solicitud;
        $arr_respuestas     =   $this->getRespuestas($id_solicitud);
        $satisfaccion       =   $this->getSatisfaccion($id_solicitud);

        $data   =   Solicitud::join('gsut_estado as estado','estado.id','=','gsut_solicitud.id_estado')
                    ->leftjoin('gsut_vecino as vecino','vecino.nr_rut','=','gsut_solicitud.nr_rut_vecino')
                    ->join('gsut_prioridad as prioridad','prioridad.id','=','gsut_solicitud.id_prioridad')
                    ->join('gsut_tipo as tipo','tipo.id','=','gsut_solicitud.id_tipo')
                    ->select(
                        'tipo.nm_tipo',
                        'gsut_solicitud.id_direccion',
                        'gsut_solicitud.tp_feedback_vecino',
                        'gsut_solicitud.str_feedback_vecino',
                        'gsut_solicitud.tp_satisfaccion_vecino',
                        'gsut_solicitud.nr_ticket',
                        'gsut_solicitud.tp_feedback',
                        'gsut_solicitud.fc_feedback',
                        'gsut_solicitud.str_feedback',
                        'gsut_solicitud.tp_recepcion',
                        'gsut_solicitud.tp_satisfaccion',
                        'gsut_solicitud.id_direccion_destino',
                        'gsut_solicitud.id_estado as id_estado_solicitud',
                        'prioridad.id as id_prioridad_solicitud',
                        'prioridad.nm_prioridad as nm_prioridad_solicitud',
                        DB::raw('concat(vecino.nm_vecino," ",vecino.ap_paterno," ",vecino.ap_materno) as nm_vecino'),
                        'gsut_solicitud.str_observacion as str_sol_observacion',
                        'gsut_solicitud.str_observacion_html as str_solicitud_html',
                        'gsut_solicitud.tp_visado_mt',
                        'gsut_solicitud.tp_rechazado')
                    ->selectRaw('DATE_FORMAT(gsut_solicitud.fc_creacion,"%d-%m-%Y") as fc_creacion')
                    ->selectRaw('DATE_FORMAT(gsut_solicitud.fc_creacion," %h:%i") as hr_creacion')
                    ->selectRaw('DATE_FORMAT(gsut_solicitud.fc_feedback_vecino,"%d-%m-%Y %h:%i") as fc_feedback_vecino')
                    ->selectRaw("(case 
                                when estado.id = 1 then 'Ingresada'
                                when estado.id = 2 then 'En Revisión'
                                when estado.id = 3 then 'En Revisión'
                                when estado.id = 4 then 'En Revisión'
                                when estado.id = 5 then 'Cerrada'
                                when estado.id = 6 then 'Cerrada'
                                when estado.id = 7 then 'Anulada'
                                when estado.id = 8 then 'Derivado'
                                when estado.id = 9 then 'Derivado'
                                when estado.id = 10 then 'Derivado'
                                end) as nm_estado")  
                    ->where('gsut_solicitud.id','=',$id_solicitud)
                    ->first();
        

        return ['arr_historial' => $data, 'arr_respuestas' => $arr_respuestas,'satisfaccion'=>$satisfaccion];
    }
    //------------------------------------------------------------
    public function storeSolicitud(SolicitudRequest $request)
    {
        // dd([
        //     'request'   =>  $request->all(),
        // ]);
        //-------------------------------------------------------------------------
        $server     =   $request->root();
        $id_rol     =   0;
        //-------------------------------------------------------------------------
        /*----- Datos de Vecino -----*/
        $rut_vecino         =   Rut::parse($request->rut_vecino)->format(Rut::FORMAT_WITH_DASH);
        $nm_vecino          =   strtoupper(trim($request->nm_vecino));
        $ap_paterno         =   strtoupper(trim($request->ap_paterno));
        $ap_materno         =   strtoupper(trim($request->ap_materno));
        $str_vecino         =   strtoupper(trim($request->str_vecino));
        $nm_mail            =   strtoupper(trim($request->nm_mail));
        //-------------------------------------------------------------------------
        /*----- Datos de la Solicitud -----*/
        $key_uuid           =   $request->key_uuid;
        $nr_motivo          =   $request->nr_motivo;
        $nr_tipo            =   $request->nr_tipo;
        $nm_tipo            =   $request->nm_tipo;
        $nr_dir_destino     =   $request->nr_dir_municipal;
        $nr_depto_destino   =   $request->nr_depto;
        $tp_ubicacion       =   $request->tp_ubicacion;
        $arr_dir_solicitud  =   null;
        $str_solicitud      =   $request->str_aclaratoria_solicitud;
        $str_descripcion    =   $request->str_observacion;
        $str_html           =   $request->str_observacion_html;
        $rut_vecino         =   Rut::parse($request->rut_vecino)->format(Rut::FORMAT_WITH_DASH);
        //-------------------------------------------------------------------------
        /*----- Datos Complementarios -----*/
        $nr_anio            =   Date('Y');
        $nr_mes             =   Date('m');
        $nr_dia             =   Date('d');
        $fc_actual          =   Date("Y-m-d H:i:s");
        $nr_folio           =   1;
        $nr_ticket          =   '';
        $id_autor           =   0; // 69070400-5 - Municipalidad de Las Condes
        $id_municipio       =   2; // Las Condes
        $id_direccion       =   22; // Externo
        $id_depto           =   90; // Plataforma Tecnologica
        $tp_vecino          =   1; 
        $tp_direccion       =   1; // frente
        $tp_municipal       =   1; // Contexto Municipal 
        $nr_prioridad       =   2; // 1. Urgente | 2. Normal 
        $tp_ingreso         =   $request->tp_ingreso; // 1. formulario | 6. Audiencias | 7. Redes Sociales
        $id_estado          =   1; // 1. Ingresada | 2. Revision
        $tp_visado_jmt      =   0;
        $tp_rechazado       =   0;
        $tp_activo          =   1; // 1. Activo | 0. Inactivo
        //-------------------------------------------------------------------------
        $valida             =   [];
        $nr_id              =   0;
        $obj_dir_solicitud  =   null;
        $msj                =   '';
        $nr_folio           =   $this->generarNumeroFolio($nr_anio);
        $nr_ticket          =   $this->generarNumeroTicket($nr_anio, $nr_folio);
        $arr_archivo        =   $request->arr_archivo;
        $arr_observacion    =   $request->arr_observacion;
        //-------------------------------------------------------------------------
        \Log::debug(__METHOD__ . " Ingreso Solicitud Vecino : " . print_r($request->all(), true));
        //-------------------------------------------------------------------------
        try {
            if($tp_ubicacion  == 1)
            {
                $arr_dir_solicitud  =   json_decode($request->arr_direccion_solicitud[0]);

                $obj_dir_solicitud  =   [
                    'value'                 =>  $arr_dir_solicitud->value,
                    'label'                 =>  $arr_dir_solicitud->label,
                    'nr_calle'              =>  $arr_dir_solicitud->nr_calle,
                    'nm_calle'              =>  $arr_dir_solicitud->nm_calle,
                    'str_latitud'           =>  $arr_dir_solicitud->str_latitud,
                    'str_longitud'          =>  $arr_dir_solicitud->str_longitud,
                    'cod_territorial'       =>  $arr_dir_solicitud->cod_territorial,
                    'id_macroterritorio'    =>  UtilController::getIdMacroTerritorio($arr_dir_solicitud->cod_territorial),
                    'cod_unidadvecinal'     =>  $arr_dir_solicitud->cod_unidadvecinal,
                    'id_unidadvecinal'      =>  UtilController::getIdUnidadVecinal($arr_dir_solicitud->cod_unidadvecinal),
                ];
            }
            else 
            {
                $arr_dir_solicitud  =   UtilController::getUbicacionDeptoMunicipal($nr_depto_destino);

                $obj_dir_solicitud  =   [
                    'value'                 =>  $arr_dir_solicitud->value,
                    'label'                 =>  $arr_dir_solicitud->label,
                    'nr_calle'              =>  $arr_dir_solicitud->nr_calle,
                    'nm_calle'              =>  $arr_dir_solicitud->nm_calle,
                    'str_latitud'           =>  $arr_dir_solicitud->str_latitud,
                    'str_longitud'          =>  $arr_dir_solicitud->str_longitud,
                    'cod_territorial'       =>  $arr_dir_solicitud->cod_territorial,
                    'id_macroterritorio'    =>  UtilController::getIdMacroTerritorio($arr_dir_solicitud->cod_territorial),
                    'cod_unidadvecinal'     =>  $arr_dir_solicitud->cod_unidadvecinal,
                    'id_unidadvecinal'      =>  UtilController::getIdUnidadVecinal($arr_dir_solicitud->cod_unidadvecinal),
                ];
            }                
            //-------------------------------------------------------------------------
            $info   =   [
                'key_uuid'              =>  $key_uuid,                                          // form
                'nr_anio'               =>  $nr_anio,                                           // interno
                'nr_mes'                =>  $nr_mes,                                            // interno
                'nr_dia'                =>  $nr_dia,                                            // interno
                'nr_folio'              =>  $nr_folio,                                          // interno
                'nr_ticket'             =>  $nr_ticket,                                         // interno
                'id_municipio'          =>  $id_municipio,                                      // interno
                'id_direccion'          =>  $id_direccion,                                      // interno
                'id_depto'              =>  $id_depto,                                          // interno
                'id_motivo'             =>  $nr_motivo,                                         // form
                'id_tipo'               =>  $nr_tipo,                                           // form 
                'nm_tipo'               =>  $nm_tipo,                                           // form
                'id_prioridad'          =>  $nr_prioridad,                                      // form
                'id_direccion_destino'  =>  $nr_dir_destino,                                    // form 
                'id_depto_destino'      =>  $nr_depto_destino,                                  // form 
                'id_ubicacion'          =>  $obj_dir_solicitud['value'],                        // form 
                'id_macroterritorio'    =>  $obj_dir_solicitud['id_macroterritorio'],           // form 
                'id_unidadvecinal'      =>  $obj_dir_solicitud['id_unidadvecinal'],             // form                         
                'nr_direccion'          =>  $obj_dir_solicitud['nr_calle'],                     // form 
                'nm_direccion'          =>  strtoupper(trim($obj_dir_solicitud['nm_calle'])),   // form 
                'nm_ubicacion'          =>  strtoupper(trim($obj_dir_solicitud['label'])),      // form 
                'str_latitud'           =>  $obj_dir_solicitud['str_latitud'],                  // form 
                'str_longitud'          =>  $obj_dir_solicitud['str_longitud'],                 // form 
                'tp_direccion'          =>  $tp_direccion,                                      // form 
                'nr_rut_vecino'         =>  $rut_vecino,                                        // form 
                'str_observacion'       =>  strtoupper(trim($str_descripcion)),                 // form 
                'str_observacion_html'  =>  strtoupper(trim($str_html)),                        // form 
                'str_aclaratoria'       =>  strtoupper(trim($str_solicitud)),                   // form 
                'fc_creacion'           =>  $fc_actual,                                         // interno
                'tp_visado_mt'          =>  $tp_visado_jmt,                                     // interno
                'tp_rechazado'          =>  $tp_rechazado,                                      // interno
                'id_estado'             =>  $id_estado,                                         // interno
                'contexto_municipal'    =>  $tp_municipal,                                      // form 
                'tp_ingreso'            =>  $tp_ingreso,                                        // interno
                'id_autor'              =>  $id_autor,                                          // interno
                'id_rol'                =>  $id_rol,                                            // interno
                'tp_activo'             =>  $tp_activo,                                         // Interno
                'tp_vecino'             =>  $tp_vecino,                                         // Interno
            ];
            //-------------------------------------------------------------------------
            $model  =   Solicitud::create($info);
            $nr_id  =   $model->id;
            $msj    =   'Se ha registrado la solicitud correctamente con el número de ticket <strong>#'.$nr_ticket .'</strong>'.
                        '<br> <center><i>Favor tomar nota del número de ticket</i></center>';

            if($nr_id)
            {
                GsutVecinoController::setRegistrarVecino($request);

                // Proceso de Almacenamiento de Archivos
                ArchivosController::registrarArchivo('tb_solicitud', $nr_id, $arr_archivo, $arr_observacion);

                // Proceso Ingreso Flujo
                GsutFlujoController::ingresoSolicitud($nr_id,$nr_tipo,$tp_ingreso);

                // Generar Token de Acceso para ver Ticket
                try {
                        DB::transaction(function () use (&$model, &$msj, $server) {
                            $model->createTokenTicket();

                            if($model->token_report) {
                                $msj    =   $msj . '<br> Para descargar el Ticket de Atención favor haga clic en el link de descarga 
                                                    <a target="_blank" href="'.$server.'/atencion_vecino/ticket/'.$model->token_report.'" class="btn btn-sm btn-danger">Descargar</a>';
                            }
                        });
                } catch (\Illuminate\Database\QueryException $e) {
                    $arr_valida =   [
                                        'error'     =>  true,
                                        'title'     =>  'Token ticket',
                                        'msj'       =>  'No es posible generar el token',
                                    ];
                    \Log::debug(__METHOD__ . " Token Ticket : " . print_r($arr_valida, true));
                } catch (Exception $e) {
                    $arr_valida =   [
                                        'error'     =>  true,
                                        'title'     =>  'Token ticket',
                                        'msj'       =>  'No es posible generar el token',
                                    ];
                    \Log::debug(__METHOD__ . " Token Ticket : " . print_r($arr_valida, true));
                }

                $valida     =   [
                                    'error'     =>  0,
                                    'existe'    =>  1,
                                    'tp_icon'   =>  'success',
                                    'title'     =>  'Registro de Solicitud',
                                    'msj'       =>  $msj,
                                ];
                                
                EmailController::enviarNotificacionVecino(101, $nr_id, ''); // 101. Notificación de ingreso solicitud al vecino
                EmailController::enviarNotificaciones(50, $nr_id); // 50. Notificación de ingreso de solicitud a administrador de Atención al Vecino
                BusController::storeVecino($request); // Registra y Actualiza datos de los vecinos del Bus de Integración
            }
            else
            {
                $msj        =   'No se ha podido registrar la solicitud, favor intente nuevamente.';

                $valida     =   [
                                    'error'     =>  1,
                                    'existe'    =>  0,
                                    'tp_icon'   =>  'warning',
                                    'title'     =>  'Registro de Solicitud',
                                    'msj'       =>  $msj,
                                ];
            }   
        } catch (Exception $e) {
             $valida     =   [
                                'error'     =>  1,
                                'existe'    =>  0,
                                'tp_icon'   =>  'error',
                                'title'     =>  'Registro de Solicitud',
                                'msj'       =>  'Se ha producido un problema al registrar la solicitud',
                            ];
        }

        return $valida;       
    }
    //------------------------------------------------------------
    public static function setRegistrarVecino(Request $request)
    {
        $rut_vecino         =   Rut::parse($request->rut_vecino)->format(Rut::FORMAT_WITH_DASH);
        $nm_vecino          =   strtoupper(trim($request->nm_vecino));
        $ap_paterno         =   strtoupper(trim($request->ap_paterno));
        $ap_materno         =   strtoupper(trim($request->ap_materno));
        $str_vecino         =   trim($nm_vecino . ' ' . $ap_paterno . ' ' . $ap_materno); // strtoupper(trim($request->str_vecino));
        $nr_prefijo         =   $request->pre_fijo;
        $nr_telefono        =   $request->nr_telefono;
        $nm_mail            =   strtoupper(trim($request->nm_mail));
        $arr_dir_vecino     =   null;
        $obj_dir_vecino     =   (Object) [];
        $tp_direccion       =   $request->tp_ubicacion_vecino;
        $str_aclaratoria    =   $request->str_aclaratoria_vecino;
        $nr_region          =   13; // RM
        $nr_comuna          =   271; // Las Condes
        $tp_comuna          =   ($nr_comuna == $request->nr_comuna) ? 1 : 0; // $request->tp_comuna;
        $nr_autor           =   @Auth::user()->id?:0;
        $tp_vecino          =   0; // (count(RestController::ConsultaTajertaVecino($rut_vecino))) ? 1 : 0;
        $tp_activo          =   1;
        $str_titulo         =   'Registro de Vecino';
        $str_msj            =   'Se ha registrado al vecino';
        $valida             =   [];

        try {
            // Datos Direccion Vecino
            if($tp_comuna == 1)
            {
                if (is_string($request->arr_ubicacion_vecino) && UtilController::isJson($request->arr_ubicacion_vecino))
                {
                    $arr_vecino     =   json_decode($request->arr_ubicacion_vecino, true);
                    $arr_dir_vecino =   (Object) $arr_vecino[0];
                }
                else
                    $arr_dir_vecino  =   json_decode($request->arr_ubicacion_vecino[0]);
                
                $obj_dir_vecino  =   (Object) [
                    'value'                 =>  $arr_dir_vecino->value,
                    'label'                 =>  $arr_dir_vecino->label,
                    'nr_calle'              =>  $arr_dir_vecino->nr_calle,
                    'nm_calle'              =>  $arr_dir_vecino->nm_calle,
                    'str_latitud'           =>  $arr_dir_vecino->str_latitud,
                    'str_longitud'          =>  $arr_dir_vecino->str_longitud,
                    'cod_territorial'       =>  $arr_dir_vecino->cod_territorial,
                    'id_macroterritorio'    =>  UtilController::getIdMacroTerritorio($arr_dir_vecino->cod_territorial),
                    'cod_unidadvecinal'     =>  $arr_dir_vecino->cod_unidadvecinal,
                    'id_unidadvecinal'      =>  UtilController::getIdUnidadVecinal($arr_dir_vecino->cod_unidadvecinal),
                ];
            }
            else
            {
                $nr_region      =   $request->nr_region;
                $nr_comuna      =   $request->nr_comuna;
                $nm_ubicacion   =   $request->nm_ubicacion_vecino;

                $obj_dir_vecino  =   (Object) [
                    'value'                 =>  0,
                    'label'                 =>  $nm_ubicacion,
                    'nr_calle'              =>  0,
                    'nm_calle'              =>  '',
                    'str_latitud'           =>  '',
                    'str_longitud'          =>  '',
                    'cod_territorial'       =>  '',
                    'id_macroterritorio'    =>  0,
                    'cod_unidadvecinal'     =>  '',
                    'id_unidadvecinal'      =>  0,
                ];
            }

            $dataVecino =   [
                'nr_rut'                =>  $rut_vecino,
                'nm_vecino'             =>  $nm_vecino,                
                'ap_paterno'            =>  $ap_paterno,
                'ap_materno'            =>  $ap_materno,
                'id_region'             =>  $nr_region,
                'id_comuna'             =>  $nr_comuna,
                'id_ubicacion'          =>  $obj_dir_vecino->value,
                'id_macroterritorio'    =>  $obj_dir_vecino->id_macroterritorio,
                'id_unidadvecinal'      =>  $obj_dir_vecino->id_unidadvecinal,
                'nr_calle'              =>  $obj_dir_vecino->nr_calle,
                'nm_calle'              =>  $obj_dir_vecino->nm_calle,
                'nm_ubicacion'          =>  $obj_dir_vecino->label,
                'str_latitud'           =>  $obj_dir_vecino->str_latitud,
                'str_longitud'          =>  $obj_dir_vecino->str_longitud,
                'tp_direccion'          =>  $tp_direccion,
                'str_aclaratoria'       =>  $str_aclaratoria,
                'nr_prefijo'            =>  $nr_prefijo,
                'nr_telefono'           =>  $nr_telefono,
                'nm_mail'               =>  $nm_mail,
                'tp_vecino'             =>  $tp_vecino,
                'id_autor'              =>  $nr_autor,
                'tp_activo'             =>  $tp_activo
            ];

            $existe_vecino     = Vecino::where('nr_rut','=',$rut_vecino)->count();

            if($existe_vecino)
            {
                Vecino::where('nr_rut','=',$rut_vecino)->update($dataVecino);
                $str_titulo =   'Actualización de Vecino';
                $str_msj    =   'Se han actualizado los datos del vecino';
            }
            else
            {
                Vecino::create($dataVecino);
                $str_titulo =   'Registro de Vecino';
                $str_msj    =   'Se ha registrado al vecino : "' . $str_vecino . '"';
            }
           
            $valida =   [
                'error'     =>  0,
                'existe'    =>  1,
                'tp_icon'   =>  'success',
                'title'     =>  $str_titulo,
                'msj'       =>  $str_msj,
            ];
        } catch (Exception $e) {
            $arr_error      =   [
                                    'exception'=>$e->getMessage(),
                                    'toString'=>$e->__toString(),
                                ];
            \Log::error(__METHOD__ . " Exception : " . print_r($arr_error, true));
            
            $valida     =   [
                                'error'     =>  1,
                                'existe'    =>  0,
                                'tp_icon'   =>  'error',
                                'title'     =>  $str_titulo,
                                'msj'       =>  'Ha ocurrido un error al registrar al vecino : "' .$str_vecino . '"'
                            ];
        }

        return $valida;
    }
    //------------------------------------------------------------
    public function generarNumeroFolio($nr_anio)
    {
        $nr_folio   =   1;

        try {
            $data   =   Solicitud::
                        select('nr_folio')
                        ->where('nr_anio','=',$nr_anio)
                        ->orderBy('nr_folio','DESC')
                        ->get()->take(1);

            if($data->count())
                $nr_folio   =   $data->first()->nr_folio + 1;
        } catch (Exception $e) {
            $nr_folio   =   1;
        }

        return $nr_folio;
    }
    //------------------------------------------------------------
    public function generarNumeroTicket($nr_anio, $nr_folio)
    {
        $str_zero       =   '0';
        $str_folio      =   str_pad($nr_folio, 4, $str_zero, STR_PAD_LEFT);
        $str_ticket     =   $nr_anio.$str_folio;

        return  $str_ticket;
    }
    //------------------------------------------------------------
    public function buscarSolicitud(Request $request)
    {
        $nr_rut     =   Rut::parse($request->nr_rut)->quiet()->validate();

        if(!$nr_rut)
            return [
                'error'     =>  1,
                'msj'       =>  'El rut ingresado es inválido',
                'nr_rut'    =>  $nr_rut,
            ];

        $nr_rut         =   Rut::parse($request->nr_rut)->format(Rut::FORMAT_WITH_DASH);
        $fl_mostrar     =   $request->fl_mostrar;
        $page           =   $request->page;

        $data           =   Solicitud:: 
                            join('gsut_prioridad as prioridad','prioridad.id','=','gsut_solicitud.id_prioridad')
                            //->join('gsut_motivo as motivo','motivo.id','=','gsut_solicitud.id_motivo')
                            ->join('gsut_tipo as tipo','tipo.id','=','gsut_solicitud.id_tipo')
                            ->join('gsut_estado as estado','estado.id','=','gsut_solicitud.id_estado')
                            ->join('gsut_direccion as dir_destino','dir_destino.id','=','gsut_solicitud.id_direccion_destino')
                            ->join('gsut_depto as depto_destino','depto_destino.id','=','gsut_solicitud.id_depto_destino')
                            ->leftjoin('gsut_macroterritorio as mt','mt.id','=','gsut_solicitud.id_macroterritorio')
                            ->leftjoin('gsut_unidadvecinal as uv','uv.id','=','gsut_solicitud.id_unidadvecinal')
                            ->join('gsut_tipo_ingreso as ti','ti.id','=','gsut_solicitud.tp_ingreso')
                            ->select('gsut_solicitud.id',
                                'gsut_solicitud.nr_ticket',
                                'gsut_solicitud.id_municipio',
                                'gsut_solicitud.id_direccion',
                                'gsut_solicitud.id_depto',
                                'gsut_solicitud.id_tipo',
                                'gsut_solicitud.nm_tipo',
                                'gsut_solicitud.id_prioridad',
                                //'gsut_solicitud.id_motivo',
                                //'motivo.nm_motivo',
                                'gsut_solicitud.id_direccion_destino',
                                'dir_destino.nm_direccion as nm_direccion_destino',
                                'gsut_solicitud.id_depto_destino',
                                'depto_destino.nm_depto as nm_depto_destino',
                                'gsut_solicitud.id_ubicacion',
                                'gsut_solicitud.id_macroterritorio',
                                'mt.cod_territorio',
                                'gsut_solicitud.id_unidadvecinal',
                                'uv.cod_unidadvecinal',
                                'uv.nm_unidadvecinal',
                                'gsut_solicitud.tp_direccion',
                                'gsut_solicitud.nr_direccion',
                                'gsut_solicitud.nm_direccion',
                                'gsut_solicitud.nm_ubicacion',
                                'gsut_solicitud.str_latitud',
                                'gsut_solicitud.str_longitud',
                                'gsut_solicitud.nr_rut_vecino',
                                'gsut_solicitud.str_observacion',
                                'gsut_solicitud.str_observacion_html as str_solicitud_html',
                                'gsut_solicitud.str_aclaratoria',
                                'gsut_solicitud.tp_visado_mt',
                                'gsut_solicitud.tp_rechazado',
                                'gsut_solicitud.id_estado',
                                'estado.nm_color',
                                //'gsut_solicitud.contexto_municipal',
                                'gsut_solicitud.tp_ingreso',
                                'ti.nm_ingreso',
                                'gsut_solicitud.id_autor',
                                'gsut_solicitud.tp_activo',
                                'gsut_solicitud.tp_feedback',
                                'gsut_solicitud.tp_recepcion',
                                'gsut_solicitud.tp_satisfaccion',
                                'gsut_solicitud.str_feedback',
                                'gsut_solicitud.tp_feedback_vecino',
                                'gsut_solicitud.fc_feedback_vecino',
                                'gsut_solicitud.tp_satisfaccion_vecino',
                                'gsut_solicitud.str_feedback_vecino',
                                'gsut_solicitud.fc_creacion as fc_ingreso',
                                'gsut_solicitud.fc_publicado',
                                'gsut_solicitud.created_at',
                                'gsut_solicitud.updated_at as fc_tramitacion',
                                DB::raw('0 as territorial'))
                            ->selectRaw("(case 
                                            when estado.id = 1 then 'Ingresada'
                                            when estado.id = 2 then 'En Revisión'
                                            when estado.id = 3 then 'En Revisión'
                                            when estado.id = 4 then 'En Revisión'
                                            when (estado.id = 5 && gsut_solicitud.tp_feedback_vecino = 0) then 'Cerrado sin Calificación'
                                            when (estado.id = 5 && gsut_solicitud.tp_feedback_vecino = 1) then 'Cerrado con Calificación'
                                            when (estado.id = 6 && gsut_solicitud.tp_feedback_vecino = 0) then 'Cerrado sin Calificación'
                                            when (estado.id = 6 && gsut_solicitud.tp_feedback_vecino = 1) then 'Cerrado con Calificación'
                                            when estado.id = 7 then 'Anulada'
                                            when estado.id = 8 then 'Derivado'
                                            when estado.id = 9 then 'Derivado'
                                            when estado.id = 10 then 'Derivado'
                                            end) as nm_estado") 
                            ->where('gsut_solicitud.nr_rut_vecino','=',$nr_rut)
                            ->where('gsut_solicitud.tp_activo','=',1)
                            ->orderBy('gsut_solicitud.created_at','DESC');

        /*---------- Inicio - Filtro ----------*/
            if($fl_mostrar != 0)
            {
                if($fl_mostrar == 3)
                    $data   =   $data->whereIn('gsut_solicitud.id_estado',[1,2,3,4]);
                if($fl_mostrar == 5)
                    $data   =   $data->whereIn('gsut_solicitud.id_estado',[5,6]);
                if($fl_mostrar != 3 && $fl_mostrar != 5)
                    $data   =   $data->where('gsut_solicitud.id_estado','=',$fl_mostrar);
            }
        /*---------- Termino - Filtro ----------*/
        $data   =   $data->groupBy('gsut_solicitud.id');

        $data   =   $data->paginate(20);

        $data->getCollection()->transform(function ($value) {
            $value->tiempo_tramitado    =   UtilController::getTiempoTramitacion($value->fc_ingreso, $value->fc_publicado, $value->id_estado);
            $value->tiene_pp            =   (UtilController::getCantidadRespuestas(2, $value->id) != 0) ? 'SI' : 'NO'; // Tiene Publicaciones Parciales                
            return $value;
        });

        return [
            'pagination' => [
                'total'        => $data->total(),
                'current_page' => $data->currentPage(),
                'per_page'     => $data->perPage(),
                'last_page'    => $data->lastPage(),
                'from'         => $data->firstItem(),
                'to'           => $data->lastItem(),
            ],
            'info'             => $data,
            'rut_vecino'       => $nr_rut,
        ];
    }
    //------------------------------------------------------------
    private function validarDatos(Request $request)
    {
        $customMessages = [
            'required'          =>  'El campo :attribute es requerido.',
            'numeric'           =>  'El campo :attribute debe ser númerico.',
            'size'              =>  'El campo :attribute debe tener :size caracteres',
            'key_uuid.required' =>  'Se requiere un identificador único para la solicitud',
            'key_uuid.uuid'     =>  'El identificador debe ser un UUID válido',
            'key_uuid.unique'   =>  'Esta solicitud ya fue procesada anteriormente',
        ];

        $attributes = [
            'key_uuid'          =>  'Identificador Único',
            'rut_vecino'        =>  'Rut',
            'nm_vecino'         =>  'Nombre',
            'ap_paterno'        =>  'Primer Apellido',
            'ap_materno'        =>  'Segundo Apellido',
            'nm_mail'           =>  'Correo',
            // 'password'          =>  'Password',
            'nr_telefono'       =>  'Teléfono',
            'nr_region'         =>  'Región',
            'nr_comuna'         =>  'Comuna',
            'nm_ubicacion_vecino'   =>  'Dirección del Vecino',

            'arr_ubicacion_vecino'                      =>  'Dirección del Vecino',
            'arr_ubicacion_vecino.value'                =>  'Código Ubicación',
            'arr_ubicacion_vecino.label'                =>  'Dirección Ubicación',
            'arr_ubicacion_vecino.nr_calle'             =>  'Numeración Ubicación',
            'arr_ubicacion_vecino.nm_calle'             =>  'Dirección Ubicación Solicitud',
            'arr_ubicacion_vecino.str_latitud'          =>  'Latitud',
            'arr_ubicacion_vecino.str_longitud'         =>  'Longitud',
            'arr_ubicacion_vecino.cod_territorial'      =>  'Código Macrozona',
            'arr_ubicacion_vecino.cod_unidadvecinal'    =>  'Código Unidad Vecinal',   
        ];

        $rules  =   [
            // 'key_uuid'          =>  [
            //     'required',
            //     'uuid',
            //     Rule::unique('gsut_solicitud', 'key_uuid')->whereNotNull('key_uuid')
            // ],
            'rut_vecino'        =>  'required|cl_rut',
            'nm_vecino'         =>  'required|string|min:2',
            'ap_paterno'        =>  'required|string|min:2',
            'ap_materno'        =>  'required|string|min:2',
            'nm_mail'           =>  'required|string',
            // 'password'          =>  'required|string|min:6',
            'nr_telefono'       =>  'required|string|size:8',          
            'tp_ubicacion_vecino'   =>  'required|numeric|min:1',

            'tp_comuna'         =>  'required|numeric',
            'nr_region'         =>  'required_if:tp_comuna,==,0|numeric|min:1',
            'nr_comuna'         =>  'required_if:tp_comuna,==,0|numeric|min:1',
            'nm_ubicacion_vecino'   =>  'required_if:tp_comuna,==,0|string|min:5',

            'arr_ubicacion_vecino'  =>  'required_if:tp_comuna,==,1|array|min:1',
            'arr_ubicacion_vecino.[*]["value"]'              =>  'required_if:tp_comuna,==,1|numeric',
            'arr_ubicacion_vecino.[*]["label"]'              =>  'required_if:tp_comuna,==,1|string|min:3',
            'arr_ubicacion_vecino.[*]["nr_calle"]'           =>  'required_if:tp_comuna,==,1|numeric',
            'arr_ubicacion_vecino.[*]["nm_calle"]'           =>  'required_if:tp_comuna,==,1|string|min:3',
            'arr_ubicacion_vecino.[*]["str_latitud"]'        =>  'required_if:tp_comuna,==,1|string',
            'arr_ubicacion_vecino.[*]["str_longitud"]'       =>  'required_if:tp_comuna,==,1|string',
            'arr_ubicacion_vecino.[*]["cod_territorial"]'    =>  'required_if:tp_comuna,==,1|string',
            'arr_ubicacion_vecino.[*]["cod_unidadvecinal"]'  =>  'required_if:tp_comuna,==,1|string',
        ];

        $validator  =   Validator::make($request->all(), $rules, $customMessages, $attributes);

        if ($validator->fails())
            return [
                'error'     =>  1,
                'title'     =>  'Validar Datos',
                'msj'       =>  'Favor revisar los campos requeridos',
                'validar'   =>  $validator->errors()
            ];

        return [
            'error'     =>  0,
            'title'     =>  'Validar Datos',
            'msj'       =>  'Campos requeridos validados',
            'validar'   =>  'Ok'
        ];
    }
    //------------------------------------------------------------
    public function getSolicitudesSistemaTerritorial(Request $request)
    {
        $str_buscar     =   Session::get('clave-unica')->rut;
        $fl_mostrar     =   $request->fl_mostrar;
        $page           =   $request->page;

        $data   =   SolicitudTerritorial:: 
                    join('gsut_prioridad as prioridad','prioridad.id','=','gsut_solicitud.id_prioridad')
                    //->join('gsut_motivo as motivo','motivo.id','=','gsut_solicitud.nr_motivo')
                    ->join('gsut_tipo as tipo','tipo.id','=','gsut_solicitud.id_tipo')
                    ->join('gsut_estado as estado','estado.id','=','gsut_solicitud.id_estado')
                    ->join('gsut_direccion as dir_destino','dir_destino.id','=','gsut_solicitud.id_direccion_destino')
                    ->leftjoin('gsut_macroterritorio as mt','mt.id','=','gsut_solicitud.id_macroterritorio')
                    ->leftjoin('gsut_unidadvecinal as uv','uv.id','=','gsut_solicitud.id_unidadvecinal')
                    ->select('gsut_solicitud.id',
                        'gsut_solicitud.nr_ticket',
                        'gsut_solicitud.id_municipio',
                        'gsut_solicitud.id_direccion',
                        'gsut_solicitud.id_depto',
                        'gsut_solicitud.id_tipo',
                        'gsut_solicitud.nm_tipo',
                        'gsut_solicitud.id_prioridad',
                        //'gsut_solicitud.nr_motivo as id_motivo',
                        //'motivo.nm_motivo',
                        'gsut_solicitud.id_direccion_destino',
                        'dir_destino.nm_direccion as nm_direccion_destino',
                        DB::raw('0 as id_depto_destino'),
                        DB::raw('"" as nm_depto_destino'),
                        'gsut_solicitud.id_ubicacion',
                        'gsut_solicitud.id_macroterritorio',
                        'mt.cod_territorio',
                        'gsut_solicitud.id_unidadvecinal',
                        'uv.cod_unidadvecinal',
                        'uv.nm_unidadvecinal',
                        'gsut_solicitud.tp_direccion',
                        'gsut_solicitud.nr_direccion',
                        'gsut_solicitud.nm_direccion',
                        'gsut_solicitud.nm_ubicacion',
                        'gsut_solicitud.str_latitud',
                        'gsut_solicitud.str_longitud',
                        'gsut_solicitud.nr_rut_vecino',
                        'gsut_solicitud.str_observacion',
                        'gsut_solicitud.str_observacion as str_solicitud_html',
                        DB::raw('"" as str_aclaratoria'),
                        'gsut_solicitud.tp_visado_mt',
                        'gsut_solicitud.tp_rechazado',
                        'gsut_solicitud.id_estado',
                        DB::raw('"" as nm_color'),
                        //'gsut_solicitud.tp_municipal as contexto_municipal',
                        DB::raw('1 as tp_ingreso'),
                        DB::raw('"Formulario" as nm_ingreso'),
                        'gsut_solicitud.id_autor',
                        'gsut_solicitud.tp_activo',
                        'gsut_solicitud.tp_feedback',
                        'gsut_solicitud.tp_recepcion',
                        'gsut_solicitud.tp_satisfaccion',
                        'gsut_solicitud.str_feedback',
                        DB::raw('0 as tp_feedback_vecino'),
                        DB::raw('null as fc_feedback_vecino'),
                        DB::raw('0 as tp_satisfaccion_vecino'),
                        DB::raw('"" as str_feedback_vecino'),
                        'gsut_solicitud.created_at as fc_ingreso',
                        'gsut_solicitud.updated_at as fc_tramitacion',
                        DB::raw('1 as territorial'))
                    ->selectRaw("(case 
                                    when estado.id = 1 then 'Ingresada'
                                    when estado.id = 2 then 'En Revisión'
                                    when estado.id = 3 then 'En Revisión'
                                    when estado.id = 4 then 'Rechazado'
                                    when estado.id = 5 then 'En Revisión'
                                    when estado.id = 6 then 'En Revisión'
                                    when estado.id = 7 then 'Rechazado'
                                    when estado.id = 8 then 'En Revisión'
                                    when estado.id = 9 then 'En Revisión'
                                    when estado.id = 10 then 'En Revisión'
                                    when estado.id = 11 then 'En Revisión'
                                    when estado.id = 12 then 'Cerrado' 
                                    when estado.id = 13 then 'Contexto No Municipal' 
                                    end) as nm_estado") 
                    //->where('gsut_solicitud.tp_activo','=',1)
                    ->where('gsut_solicitud.nr_rut_vecino',$nr_rut)
                    ->orderBy('gsut_solicitud.created_at','DESC');

        /*---------- Inicio - Filtro ----------*/
            if($fl_mostrar != 0)
            {
                if($fl_mostrar == 3)
                    $data   =   $data->whereIn('gsut_solicitud.id_estado',[1,2,3,5,6,8,9,10,11]);
                if($fl_mostrar == 5)
                    $data   =   $data->whereIn('gsut_solicitud.id_estado',[12]);
                if($fl_mostrar != 3 && $fl_mostrar != 5)
                    $data   =   $data->where('gsut_solicitud.id_estado','=',$fl_mostrar);
            }
        /*---------- Termino - Filtro ----------*/

        $data   =   $data->paginate(20);

        $data->getCollection()->transform(function ($value) {
            $value->tiempo_tramitado    =   UtilController::getTiempoTramitacionTerritorial($value->fc_ingreso, $value->fc_tramitacion, $value->id_estado);
            return $value;
        });

        return [
            'pagination' => [
                'total'        => $data->total(),
                'current_page' => $data->currentPage(),
                'per_page'     => $data->perPage(),
                'last_page'    => $data->lastPage(),
                'from'         => $data->firstItem(),
                'to'           => $data->lastItem(),
            ],
            'info'             => $data,
            'rut_vecino'       => $str_buscar,
        ];
    }
    //------------------------------------------------------------
}
